<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Headers: *");
header("Access-Control-Allow-Methods: POST");

// ==========================================
// FILE TARGET
// ==========================================
$file = __DIR__ . "/products.json";

if (!file_exists($file)) {
    file_put_contents($file, json_encode([], JSON_PRETTY_PRINT));
}

$products = json_decode(file_get_contents($file), true);
if (!is_array($products)) $products = [];

// ==========================================
// READ INPUT
// ==========================================
$raw = file_get_contents("php://input");
$req = json_decode($raw, true);

if (!$req || empty($req["name"])) {
    echo json_encode([
        "status" => false,
        "message" => "Product name is required."
    ]);
    exit;
}

// ==========================================
// NORMALIZE FIELDS
// ==========================================
$id       = isset($req["id"]) ? intval($req["id"]) : null;
$name     = trim($req["name"]);
$category = trim($req["category"] ?? "");
$sub      = trim($req["sub"] ?? "");
$price    = intval($req["price"] ?? 0);
$stock    = intval($req["stock"] ?? 0);
$description = trim($req["description"] ?? "");

// Images array (dashboard sends "images")
$images = isset($req["images"]) && is_array($req["images"])
    ? array_values(array_filter(array_map("trim", $req["images"])))
    : [];

// VARIANTS (dashboard sends inside: {variants:{colors:[],sizes:[]}})
$variants = $req["variants"] ?? [];

$colors = [];
if (isset($variants["colors"]) && is_array($variants["colors"])) {
    foreach ($variants["colors"] as $c) {
        if (!empty($c["name"]) && !empty($c["hex"])) {
            $colors[] = [
                "name" => trim($c["name"]),
                "hex"  => trim($c["hex"])
            ];
        }
    }
}

$sizes = [];
if (isset($variants["sizes"]) && is_array($variants["sizes"])) {
    $sizes = array_values(array_filter(array_map("trim", $variants["sizes"])));
}

// ==========================================
// UPDATE PRODUCT
// ==========================================
if ($id !== null) {

    $found = false;

    foreach ($products as &$p) {
        if ($p["id"] == $id) {

            $p["name"]     = $name;
            $p["category"] = $category;
            $p["sub"]      = $sub;
            $p["price"]    = $price;
            $p["stock"]    = $stock;

            // Correct keys -> match dashboard!
            $p["images"]   = $images;
            $p["variants"] = [
                "colors" => $colors,
                "sizes"  => $sizes
            ];
            
            $p["description"] = $description;

            $found = true;
            break;
        }
    }

    if (!$found) {
        echo json_encode([
            "status" => false,
            "message" => "Product not found."
        ]);
        exit;
    }

    file_put_contents($file, json_encode($products, JSON_PRETTY_PRINT));

    echo json_encode([
        "status" => true,
        "message" => "Product updated.",
        "products" => $products
    ]);
    exit;
}

// ==========================================
// CREATE PRODUCT
// ==========================================
$newId = count($products) > 0
    ? max(array_column($products, "id")) + 1
    : 1;

$newProd = [
    "id"       => $newId,
    "name"     => $name,
    "category" => $category,
    "sub"      => $sub,
    "price"    => $price,
    "stock"    => $stock,
    
    "description" => $description,

    // Correct keys
    "images"   => $images,
    "variants" => [
        "colors" => $colors,
        "sizes"  => $sizes
    ]
];

$products[] = $newProd;

file_put_contents($file, json_encode($products, JSON_PRETTY_PRINT));

echo json_encode([
    "status" => true,
    "message" => "Product added.",
    "product" => $newProd,
    "products" => $products
]);
