<?php
header("Content-Type: application/json");
header("Access-Control-Allow-Origin: *"); 
header("Access-Control-Allow-Headers: *");

$ordersFile = __DIR__ . "/orders.json";

// ===============================
// 1. Terima JSON Payload
// ===============================
$raw = file_get_contents("php://input");
$data = json_decode($raw, true);

if (!$data) {
    echo json_encode(["status" => false, "message" => "Invalid payload"]);
    exit;
}

// Validasi minimal
if (!isset($data["payment_id"]) || !isset($data["items"]) || !isset($data["total"])) {
    echo json_encode(["status" => false, "message" => "Missing required fields"]);
    exit;
}

// ===============================
// 2. Generate Order ID & Tracking ID
// ===============================
$orderId = "NR-" . date("Ymd") . "-" . rand(100000, 999999);
$trackingId = "TRK-" . rand(100000, 999999);

// ===============================
// 3. Normalisasi ITEMS
// ===============================
$normalizedItems = [];
foreach ($data["items"] as $item) {
    $normalizedItems[] = [
        "id"               => $item["id"],
        "name"             => $item["name"],
        "quantity"         => $item["quantity"],
        "price"            => $item["price"],

        // VARIANT
        "variant_color"     => $item["selectedColor"]["name"] ?? null,
        "variant_color_hex" => $item["selectedColor"]["hex"] ?? null,
        "variant_size"      => $item["selectedSize"] ?? null,

        // GAMBAR
        "img"               => $item["img"] ?? null
    ];
}

// ===============================
// 4. Siapkan Data Order
// ===============================
$newOrder = [
    "order_id"     => $orderId,
    "payment_id"   => $data["payment_id"],
    "tracking_id"  => $trackingId,
    "created_at"   => date("Y-m-d H:i:s"),
    "customer"     => $data["customer"] ?? [],
    "items"        => $normalizedItems,
    "subtotal"     => $data["subtotal"] ?? 0,
    "tax"          => $data["tax"] ?? 0,
    "total"        => $data["total"],
    "status"       => "paid"
];

// ===============================
// 5. Write File Safely
// ===============================
if (!file_exists($ordersFile)) {
    file_put_contents($ordersFile, json_encode([], JSON_PRETTY_PRINT));
}

$fp = fopen($ordersFile, "c+");

if (flock($fp, LOCK_EX)) {
    $existing = stream_get_contents($fp);
    $orders = json_decode($existing, true);
    if (!is_array($orders)) $orders = [];

    $orders[] = $newOrder;

    ftruncate($fp, 0);
    rewind($fp);
    fwrite($fp, json_encode($orders, JSON_PRETTY_PRINT));

    flock($fp, LOCK_UN);
}

fclose($fp);

echo json_encode([
    "status" => true,
    "order_id" => $orderId,
    "tracking_id" => $trackingId
]);
